//*************************************************************************************************
//
//	Description:
//		Unlit - temp shader for simple single diffuse texture material with no lighting
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		MHobbs			    11/01/2006  0.1						Created
//		TNettleship			11/20/2006  0.2						Added translucent techniques
//		TMann						11/24/2006	1.2						Added _WIN_GL_ support
//		TMann						11/27/2006	1.3						Texture filtering for GL
//		TMann						12/11/2006	1.4						PS3 version
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship     07/24/2007  2.01          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.02					Removed texture bias.
//		TNettleship			10/23/2007	2.03					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"
#include "lighting_globals.fxh"



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;


float4 textureScrollSpeed	: SCROLLSPEED
<
	string UIName = "scroll speed";
	string UIWidget = "Spinner";
	bool appEdit = true;
	bool export = true;
>;

float time
<
	string UIName = "time";
	bool appEdit = true;
	bool export = true;
>;


//
// Transforms
//
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Second UV channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTextureLayer1 : DiffuseMap											// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTextureLayer1 : TEXTURE											// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture 1";
	bool appEdit = true;
>;

#ifdef _3DSMAX_
texture diffuseTextureLayer2 : DiffuseMap											// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTextureLayer2 : TEXTURE											// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture 2";
	bool appEdit = true;
>;

float4 staticColour : Color												// static colour multiplier
<
	string UIName = "Static Colour";
	string UIType = "ColorSwatch";
	bool appEdit = true;
> = {1.0f, 1.0f, 1.0f, 1.0f};




//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMapLayer1 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTextureLayer1"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTextureLayer1 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D diffuseMapLayer2 : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTextureLayer2"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTextureLayer2 >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Functions
//




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 Norm: NORMAL;
};


// Output structure
struct VSOUTPUT
{
	float4 position	: POSITION;														// View-coords position
	float4 colour		: TEXCOORD2;															// Vertex colour
	float2 texCoord0	: TEXCOORD0;													// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;													// UV coords for texture channel 0
	
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT UnlitVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	float4x4	worldviewproj = mul( world, viewproj );

	// Copy simple invariant params to output structure
	_output.colour	 = _input.colour;
	
	
	//_output.texCoord = _input.texCoord;
	_output.texCoord0.x = _input.texCoord.x + (textureScrollSpeed.x * time );
	_output.texCoord0.y = _input.texCoord.y + (textureScrollSpeed.z * time);
	
	_output.texCoord1.x = _input.texCoord.x + (textureScrollSpeed.z * time);
	_output.texCoord1.y = _input.texCoord.y + (textureScrollSpeed.x * time);
	
	float2 offset;
	offset.x = worldCameraPos.x;
	offset.y = 0.0f - worldCameraPos.z;
	
	
	// bottom texture
	_output.texCoord0 += offset / 3000.0f;  // 8000.0f;
	
	//top texture
	_output.texCoord1 += offset / 10000.0f;  // 16000.0f;
	
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
struct PSINPUT
{
	float4 colour		: TEXCOORD2;													// Vertex colour
	float2 texCoord0	: TEXCOORD0;										// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;										// UV coords for texture channel 1
	
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT UnlitFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColourLayer1 = tex2D( diffuseMapLayer1, _input.texCoord0 );
	float4 diffuseTexColourLayer2 = tex2D( diffuseMapLayer2, _input.texCoord1 );

	float blend = diffuseTexColourLayer1.a;
	
	float4 texColour = lerp( diffuseTexColourLayer2, diffuseTexColourLayer1, blend );
	
	// Calculate pixel colour
	float4 accumulatedColour = staticColour * texColour * _input.colour;
	

	accumulatedColour.w = texColour.w;
	APPLY_DISTANCE_FOG( accumulatedColour, 0.65f );
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Unlit_Trans
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Unlit_Trans";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = false;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = false;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx UnlitVertexShader();
		PixelShader = compile sce_fp_rsx UnlitFragmentShader();
#else		
		VertexShader = compile vs_3_0 UnlitVertexShader();
		PixelShader = compile ps_3_0 UnlitFragmentShader();
#endif		
	}
}
